//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System.Diagnostics;
using System.Reflection;

using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents a strategy for retrieving parameter from an object property.
	/// </summary>
    public class PropertyParameterFactory<TObjectToCreate, TProperty, TObjectConfiguration> : ObjectFactoryBase<TObjectToCreate, TObjectConfiguration>
	{
		private readonly PropertyInfo sourceProperty;
        private readonly IObjectFactory<TObjectToCreate, TProperty> innerFactory;

		/// <summary>
		/// Initialize a new instance of the <see cref="PropertyParameterFactory"/> class with the source property and am injection strategy.
		/// </summary>
		/// <param name="sourceProperty">The <see cref="PropertyInfo"/> object used to retrieve the value to give to the injection strategy.</param>
        /// <param name="innerFactory">The strategy used to create the value.</param>
        public PropertyParameterFactory(PropertyInfo sourceProperty, IObjectFactory<TObjectToCreate, TProperty> innerFactory)
		{			
			this.sourceProperty = sourceProperty;
            this.innerFactory = innerFactory;
		}

		/// <summary>
		/// Gets the property used to retrive the value.
		/// </summary>
		/// <value>
		/// The property used to retrive the value.
		/// </value>
		public PropertyInfo SourceProperty
		{
			get { return sourceProperty; }
		}

		/// <summary>
		/// Gets the transformation strategy used to retrieve the parameter value to inject.
		/// </summary>
		/// <value>
		/// The transformation strategy used to retrieve the parameter value to inject.
		/// </value>
        public IObjectFactory<TObjectToCreate, TProperty> InnerFactory
		{
            get { return innerFactory; }
		}

        public override TObjectToCreate Create(TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
        {
            TProperty sourceValue = (TProperty)sourceProperty.GetValue(objectConfiguration, null);

            return innerFactory.Create(sourceValue, configurationSource);
        }
    }
}
